#!/usr/bin/perl
BEGIN {
    unshift(@INC,'/etc/aosi');
}
use config;
use Config::General;
use strict;
use warnings;
use utf8;
use JSON::XS;
use LWP::UserAgent;
use Data::Dumper;
use HTTP::Request;
use Net::LDAP::LDIF;
use File::Copy;
use IO::Compress::Gzip qw(gzip $GzipError);
use URI::Escape qw(uri_escape_utf8); # Added for URL encoding filters
use Try::Tiny; # Good practice for JSON decoding

BEGIN {
    unshift(@INC,$config::lib_dir);
}
use Plugins::o365connectAttributes;
use Plugins::LDIFManip;
use List::Util qw(shuffle);

my $VERSION   = 1.9; # Incremented version
my $plugin_name = 'o365connect';

my $conf = new Config::General(
    -ConfigFile      => $config::plugins_dir . '/'.$plugin_name.'.conf',
    -InterPolateVars => 1,
    -AutoLaunder     => 1
);
my %cfg = $conf->getall;

my $base_dn=$config::base_dn;
my $realm =Plugins::LDIFManip::base2realm($base_dn);
my $clientID=$cfg{'ofc_clientID'};
my $clientSecret=$cfg{'ofc_clientSecret'}; # Consider secure retrieval
my $ldif_file=$config::log_dir . '/' . $cfg{'ofc_file_prefix'} ."_". $realm . $cfg{'ofc_file_ext'};
my $archive_processed_ldif=$cfg{'ofc_archive_processed_ldif'};
#koliko sekundi plugin ceka za dodjelu licenci nakon dodavanja korisnika
our $pauza_s=1;

# --- Load Mappings (Assuming these modules are correct) ---
my %attr_map=%{$Plugins::o365connectAttributes::attr_map};
my %group_map=%{$Plugins::o365connectAttributes::group_map};
my %licence_map=%{$Plugins::o365connectAttributes::licence_map};

my $userAgent='AAI@EduHrMSGraphConnect/' . "$VERSION";

# --- Microsoft Graph API Configuration ---
# Endpoint na kojem se trazi autentikacijski token (v2.0 endpoint)
my $OauthTokenEndpoint = 'https://login.microsoftonline.com/' . $realm . '.hr/oauth2/v2.0/token';

# Microsoft Graph API Scope - resurs za kojeg se trazi token
my $graphScope = "https://graph.microsoft.com/.default";


# Endpoint za administraciju imenika putem Microsoft Graph API-a
#https://graph.microsoft.com/{version}/{tenant_id}/{resource}?query-parameters
my $serviceRootURL = 'https://graph.microsoft.com/v1.0/' . $realm . '.hr'; # v1.0 endpoint

my $authToken; # Will store the token hash { access_token => ..., expires_on => ..., token_type => ... }
my $false=JSON::XS::false;
my $true=JSON::XS::true;
my $null=undef;
$Data::Dumper::Terse = 1;
my $ua = LWP::UserAgent->new(timeout => 60); # Increased timeout slightly
$ua->agent($userAgent);

my %licences; # Cache for license details { skuPartNumber => { skuId => ..., capabilityStatus => ..., leftUnits => ...}}
my %groups;   # Cache for group details { displayName => group_id }


# --- Main Processing Logic ---
if (-e $ldif_file) {
    print my_time() . ", ";
    my $new_file=$ldif_file . "." . timestamp();
    move($ldif_file, $new_file) or die "Failed to move $ldif_file to $new_file: $!";
	print "ldif:" . $new_file . "\n";
    my $ldif = Net::LDAP::LDIF->new( "$new_file", "r", onerror => 'die');

    while (my $entry = $ldif->read_entry()) {
        next unless $entry; # Skip potential empty entries
        my $ctype=$entry->changetype();
        my $dn = $entry->dn();
        my $uid = find_uid($dn); # Extract uid part for UPN construction
        #my $upn = $uid ? lc($uid . '@' . $realm . '.hr') : undef; # Construct UPN
        # upn is hrEduPersonUniqueID if exists
        my $upn = find_upn($uid,$realm,$entry);

        print "$dn: $ctype: ";

        unless ($upn) {
             print "Cannot determine UPN from DN. Skipping.\n";
             next;
        }

        if ($ctype eq 'add') {
            # Add user returns the newly created user object (or undef on failure)
            
            #print "upn:" . $upn;
            my $new_user_data = add_user($entry, $upn);
            if ($new_user_data && $new_user_data->{id}) {
                print "Korisnik dodan u Azure AD (ID: " . $new_user_data->{id} . "). ";
                # Wait before assigning license/group
                sleep($pauza_s);
                add_licence($entry, $upn, $new_user_data->{id});
                add_user_to_group($entry, $upn, $new_user_data->{id});
            } else {
                print "Greska kod dodavanja korisnika!";
                # error_log already called inside add_user on failure
            }
        } elsif ($ctype eq 'delete') {
            # Delete requires user ID. Look it up first.
            my $user_details = get_user_details_by_upn($upn);
            if ($user_details && $user_details->{id}) {
                if (delete_user($upn, $user_details->{id})) {
                    print "OK";
                } else {
                    print "Greska kod brisanja korisnika.";
                }
            } else {
                 print "Korisnik '$upn' nije pronadjen za brisanje. Preskacem.";
                 # Potentially log this as warning via error_log if needed
            }

        } elsif ($ctype eq 'modify') {
             # Modify requires user ID. Look it up first.
             my $user_details = get_user_details_by_upn($upn);
             if ($user_details && $user_details->{id}) {
                 if (modify_user($entry, $upn, $user_details->{id})) {
                     print "OK";
                 } else {
                     print "Greska kod izmjene korisnika!";
                 }
             } else {
                 print "Korisnik '$upn' nije pronadjen za izmjenu. Preskacem.";
                 # Potentially log this as warning via error_log if needed
             }
        } else {
            print "Nedefiniran tip promjene '$ctype'. Preskacem.";
        }
        print "\n";
    }
    $ldif->done();

    if($archive_processed_ldif){
        gzip $new_file => "$new_file.gz" or warn "gzip failed for $new_file: $GzipError\n";
        # Only unlink original if gzip succeeded
        if (-e "$new_file.gz") {
             unlink $new_file or warn "Could not unlink $new_file after gzip: $!";
        }
    } else {
         unlink $new_file or warn "Could not unlink $new_file: $!";
    }

    # Clear token and caches
    $authToken=undef;
    %licences=();
    %groups=();
} 



exit 0;

# --- Subroutines ---

sub add_user {
    my ($entry, $upn) = @_;
    my $password = pwd_gen(); # Generate password

    # --- Build User Payload ---
    my %user=(
        'usageLocation'     => 'HR', # Microsoft Graph requires this for license assignment
        'accountEnabled'    => $true,
        'passwordProfile'   => {
        	'password'                      => $password,
			'forceChangePasswordNextSignIn' => $false 
        },
        # Add mapped attributes
        'userprincipalname' => $upn, # Ensure UPN is included
    );

    # Add other mapped attributes from LDIF
    foreach my $key (keys %attr_map){
        my $ldif_value = $entry->get_value($key);
         # Only add if value exists in LDIF, prevent overwriting defaults like accountEnabled
        if (defined $ldif_value && $attr_map{$key} ne 'accountEnabled' && $attr_map{$key} ne 'userPrincipalName') {
            $user{$attr_map{$key}} = $ldif_value;
        }
    }
    #print "Adding user:\n";
    #print Dumper(\%user);
    # --- Call API ---
    # POST to /users returns 201 Created and the new user object
    return call_graph_api('POST', '/users', 201, \%user);
}

# New function to get user details (including ID) by UPN
sub get_user_details_by_upn {
    my ($upn) = @_;
    my $encoded_upn = uri_escape_utf8($upn);
    # Select only necessary fields, especially the 'id'
    my $filter = "\$filter=userPrincipalName eq '$encoded_upn'&\$select=id,userPrincipalName,displayName";

    # GET /users with filter returns 200 OK and a 'value' array
    my $result = call_graph_api('GET', '/users', 200, undef, $filter);

    if ($result && $result->{value} && ref($result->{value}) eq 'ARRAY' && @{$result->{value}} > 0) {
        if (@{$result->{value}} > 1) {
            error_log("get_user_details_by_upn: Warning - Multiple users found for UPN '$upn'. Using the first one.", 0);
        }
        return $result->{value}->[0]; # Return the first user object found
    } else {
        # error_log("get_user_details_by_upn: User not found for UPN '$upn'", 0); # Optional: log not found
        return undef; # User not found or error occurred (logged in call_graph_api)
    }
}


# Function to get user by ID (replaces old get_user by DN)
sub get_user_by_id {
	my $user_id=shift;
	if($user_id) {
        # GET /users/{id} returns 200 OK
		return call_graph_api('GET', '/users/' . $user_id, 200);
	} else {
		error_log("get_user_by_id: No user ID provided.", 0);
        return undef;
	}
}

sub add_licence {
    my ($entry, $upn, $user_id) = @_;
    # Use the passed user_id, no need to find uid from DN again
    unless ($user_id) {
        error_log("add_licence: No user ID provided for UPN '$upn'. Cannot assign license.", 0);
        return 0;
    }

    my $affiliation=makni_hrvatske($entry->get_value('hrEduPersonPrimaryAffiliation'));
    #koliko puta ponovo pokusa dodati licencu
    my $retries=3;

    my @lic_to_assign;
    if(exists $licence_map{$affiliation} && ref($licence_map{$affiliation}) eq 'ARRAY') {
        @lic_to_assign = @{$licence_map{$affiliation}};
    }

    if (scalar @lic_to_assign > 0) {
        get_licences() unless %licences; # Ensure licenses are loaded
        unless (%licences) {
             error_log("add_licence: Greska u dohvatu podataka o licencama. Cannot assign license for '$upn'.", 0);
             return 0; # Critical error if we can't get license info
        }

        my $assigned = 0;
        while(my $skuPartNumber = shift @lic_to_assign) {
            unless (exists $licences{$skuPartNumber}) {
                 error_log("add_licence: Sku '$skuPartNumber' configured for affiliation '$affiliation' not found in subscribed SKUs.", 0);
                 next; # Skip this SKU
            }

            if ($licences{$skuPartNumber}{'capabilityStatus'} eq 'Enabled' && $licences{$skuPartNumber}{'leftUnits'} > 0) {
                my $params={
                  'addLicenses' => [
                                        {
                                          "disabledPlans"=> [], # Assign all service plans within the license
                                          "skuId"=> $licences{$skuPartNumber}{'skuId'}
                                        }
                                  ],
                  'removeLicenses' => []
                };

                my $i=0;
                while ($i < $retries) {
                    # POST to /users/{id}/assignLicense returns 200 OK
                    if (call_graph_api('POST', '/users/' . $user_id . '/assignLicense', 200, $params)) {
                        $licences{$skuPartNumber}{'leftUnits'}--;
                        $licences{$skuPartNumber}{'consumedUnits'}++;
                        print "Dodijeljena licenca $skuPartNumber. ";
                        $assigned = 1;
                        last; # Exit inner retry loop
                    }
                    print "\nDodjela licence '$skuPartNumber' nije uspjela ($i/$retries). Pokusavam ponovo.\n";
                    sleep($pauza_s);
                    $i++;
                 }
                 last if $assigned; # Exit outer while loop if assigned

            } else {
                error_log("add_licence: Nema preostalih omogucenih licenci tipa $skuPartNumber for '$upn'. Status: " . $licences{$skuPartNumber}{'capabilityStatus'} . ", Left: " . $licences{$skuPartNumber}{'leftUnits'}, 0);
            }
        } # end while skuPartNumber
        return $assigned;

    } else {
        print "add_licence: Za hrEduPersonPrimaryAffiliation '$affiliation' (UPN: $upn) nije podesena defaultna licenca. ";
        return 1; # Not an error, just nothing to do
    }
}

sub add_user_to_group {
    my ($entry, $upn, $user_id) = @_;
    # Use the passed user_id, no need for ADentry or find_uid
    unless ($user_id) {
        error_log("add_user_to_group: No user ID provided for UPN '$upn'. Cannot add to group.", 0);
        return 0;
    }

    my $affiliation=makni_hrvatske($entry->get_value('hrEduPersonPrimaryAffiliation'));
    #koliko puta ponovo pokusa dodati grupu
    my $retries=3;
    my @groups_to_assign;
    if(exists $group_map{$affiliation} && ref($group_map{$affiliation}) eq 'ARRAY'){
        @groups_to_assign = @{$group_map{$affiliation}};
    }

    if (scalar @groups_to_assign > 0) {
        get_groups() unless %groups; # Ensure groups cache is populated

        my $added_to_any = 0;
        while (my $group_display_name = shift @groups_to_assign) {
            # Check if group exists in our cache (means it exists in Azure AD and we have its ID)
             unless (exists $groups{$group_display_name}) {
                 # Try to create it? Or just log an error? Current script tries to create.
                 print "Grupa '$group_display_name' nije pronađena u cacheu, pokušavam kreirati...";
                 if (create_group($group_display_name)) {
                     print "Kreirana. ";
                     # Re-fetch groups to update cache - might be inefficient for many groups
                     get_groups();
                 } else {
                     error_log("add_user_to_group: Nije uspjelo kreiranje grupe '$group_display_name'. Preskačem dodavanje korisnika '$upn'.", 0);
                     next; # Skip to next group
                 }
             }

             # Check cache again after potential creation
             if (my $group_id = $groups{$group_display_name}) {
                # --- CORRECT Microsoft Graph Payload ---
                my $payload = {
                    '@odata.id' => "$serviceRootURL/users/$user_id"
                };

                my $i=0;
                while($i < $retries) {
                    # --- CORRECT Microsoft Graph Endpoint ---
                    # POST to /groups/{id}/members/$ref returns 204 No Content
                    if (call_graph_api('POST', '/groups/' . $group_id . '/members/$ref', 204, $payload)) {
                        print "Korisnik '$upn' dodan u grupu '$group_display_name'. ";
                        $added_to_any = 1;
                        last; # Exit inner retry loop
                    }
                    print "\nNije uspjelo dodavanje u grupu '$group_display_name' ($i/$retries). Pokusavam ponovo.\n";
                    sleep($pauza_s);
                    $i++;
                }
             } else {
                 error_log("add_user_to_group: Grupa '$group_display_name' još uvijek nije pronađena nakon pokušaja kreiranja. Preskačem za korisnika '$upn'.", 0);
             }
        } # end while group_display_name
        return $added_to_any;
    } else {
        print "add_user_to_group: Za hrEduPersonPrimaryAffiliation '$affiliation' (UPN: $upn) nije podesena defaultna grupa. ";
        return 1; # Not an error, just nothing to do
    }
}


sub delete_user {
    my ($upn, $user_id) = @_;
    # Use the passed user_id
    unless ($user_id) {
        error_log("delete_user: No user ID provided for UPN '$upn'. Cannot delete.", 0);
        return 0;
    }
    # DELETE /users/{id} returns 204 No Content
    return call_graph_api('DELETE', '/users/' . $user_id, 204);
}

sub modify_user {
    my ($entry, $upn, $user_id) = @_;
    # Use the passed user_id
    unless ($user_id) {
        error_log("modify_user: No user ID provided for UPN '$upn'. Cannot modify.", 0);
        return 0;
    }

    my %user_update_payload;
    my @changes = $entry->changes; # Returns array of [ op, [ [attr, [vals] ], ... ] ]

    # Process modify operations - currently only handles 'replace'
    # Example structure from Net::LDAP::Entry: [ 'replace', [ [ 'sn', ['NewSurname'] ], ['givenName', ['NewGiven']] ] ]
    while(my $operation_type=shift @changes){# e.g., 'replace', 'add', 'delete'
    	#print "cb:$operation_type\n";
         my @attribute_changes = shift @changes; # Array of [ attr, [vals] ] pairs
         if ($operation_type eq 'replace') {
             foreach my $attr_val_pair (@attribute_changes) {
                 my $ldif_attr_name = $attr_val_pair->[0];
                 my @ldif_values = @{$attr_val_pair->[1]}; # Get values array

                 if (exists $attr_map{$ldif_attr_name}) {
                     my $graph_attr_name = $attr_map{$ldif_attr_name};
                     # For 'replace', we usually take the first value for single-valued attributes
                     # Handle multi-valued attributes if necessary based on Graph schema
                     if (@ldif_values > 0) {
                         $user_update_payload{$graph_attr_name} = $ldif_values[0];
                         print " [Modifying $graph_attr_name] ";
                     } else {
                          # Replacing with empty value means clearing the attribute in Graph?
                          # Graph often uses null for this. Handle carefully.
                          $user_update_payload{$graph_attr_name} = $null;
                          print " [Clearing $graph_attr_name] ";
                     }
                 } else {
                     print " [Attribute '$ldif_attr_name' not mapped, skipping modification] ";
                 }
             }
         } elsif ($operation_type eq 'add') {
              print " [Modify operation 'add' not implemented, skipping] ";
              # TODO: Implement if needed, requires knowing if attribute is multi-valued
         } elsif ($operation_type eq 'delete') {
              print " [Modify operation 'delete' not implemented, skipping] ";
              # TODO: Implement if needed. Often means setting attribute to null or removing from array.
         }
    }

    my $size = scalar keys %user_update_payload;
    if ($size > 0) {
    	# print Dumper(\%user_update_payload);
        # PATCH /users/{id} returns 204 No Content
        return call_graph_api('PATCH', '/users/' . $user_id, 204, \%user_update_payload);
    } else {
        print " No recognized attributes to modify for $upn. ";
        return 1; # Nothing to do, consider it success
    }
}

sub get_licences {
    # Clear existing cache
    %licences = ();
    # GET /subscribedSkus returns 200 OK
    my $result = call_graph_api('GET', '/subscribedSkus', 200);
    unless ($result && $result->{value} && ref($result->{value}) eq 'ARRAY') {
         error_log("get_licences: Failed to parse license information from API response.", 0);
         return;
    }

    my @lic_list = @{$result->{value}};
    #print "Fetched " . scalar(@lic_list) . " subscribed SKUs.\n";
    while (my $licence = shift @lic_list) {
        my $skuPartNumber = $licence->{'skuPartNumber'};
        next unless $skuPartNumber; # Skip if missing identifier

        $licences{$skuPartNumber}{'skuId'} = $licence->{'skuId'};
        $licences{$skuPartNumber}{'capabilityStatus'} = $licence->{'capabilityStatus'};
        $licences{$skuPartNumber}{'consumedUnits'} = $licence->{'consumedUnits'};

        # Handle potential variations in prepaidUnits structure
        my $enabled_units = 0;
        if (exists $licence->{'prepaidUnits'} && ref($licence->{'prepaidUnits'}) eq 'HASH' && exists $licence->{'prepaidUnits'}->{'enabled'}) {
           $enabled_units = $licence->{'prepaidUnits'}->{'enabled'} || 0; # Ensure numeric
        }
        $licences{$skuPartNumber}{'prepaidUnits'} = $enabled_units;
        $licences{$skuPartNumber}{'leftUnits'} = $enabled_units - $licence->{'consumedUnits'};
    }
    # print Dumper(\%licences); # Debugging
}


sub create_group {
    my $displayName = shift;
    return 0 unless $displayName; # Need a name

    print "Attempting to create group '$displayName'... ";

    # Basic payload for a security group
    my %group_payload = (
        'displayName'     => $displayName,
        'mailEnabled'     => $false,
        'mailNickname'    => $displayName =~ s/[^a-zA-Z0-9]//gr, # Create a valid mailNickname
        'securityEnabled' => $true,
        # 'groupTypes' => [], # For M365 groups, add "Unified"
        # 'description' => "Group for $displayName", # Optional
    );

    # POST /groups returns 201 Created
    my $result = call_graph_api('POST', '/groups', 201, \%group_payload);
    if ($result && $result->{id}) {
        print "Group '$displayName' created with ID: " . $result->{id} . ". ";
        # Update local cache immediately
        $groups{$displayName} = $result->{id};
        return 1; # Success
    } else {
        print "Failed. ";
        return 0; # Failure
    }
}

sub get_groups {
    # Clear existing cache
    %groups = ();
    my $filter_parts;

    # Build the $filter string based on configured group display names
    foreach my $key (keys %group_map){
        if (ref($group_map{$key}) eq 'ARRAY') {
           foreach my $group_name (@{$group_map{$key}}) {
               push @$filter_parts, "displayName eq '" . uri_escape_utf8($group_name) . "'";
           }
        }
    }

    # Remove duplicates just in case
    my %seen;
    my @unique_filter_parts = grep { !$seen{$_}++ } @$filter_parts;

    if (scalar @unique_filter_parts == 0) {
         print "get_groups: No groups configured in group_map. Skipping fetch.";
         return;
    }

    my $filter_string = '$filter=' . join(' or ', @unique_filter_parts);
    # Also select ID
    $filter_string .= '&$select=id,displayName';

    # GET /groups with filter returns 200 OK
    my $result = call_graph_api('GET', '/groups', 200, undef, $filter_string);
    unless ($result && $result->{value} && ref($result->{value}) eq 'ARRAY') {
         error_log("get_groups: Failed to parse group information from API response.", 0);
         return;
    }

    my @group_list = @{$result->{value}};
    # print "Fetched " . scalar(@group_list) . " groups matching configuration.\n";
    while (my $group = shift @group_list) {
        # --- Store group ID using displayName as key ---
        if ($group->{displayName} && $group->{id}) {
             $groups{$group->{'displayName'}} = $group->{'id'};
        }
    }
     # print Dumper(\%groups); # Debugging
}


# Extracts the value of the 'uid' attribute from a DN string
sub find_uid {
    my $dn=shift;
    # Match uid=value, where value can contain letters, numbers, ., -, _
    if ($dn =~ /uid=([^,]+)/i) {
        my $uid=$1;
        $uid=~ s/^\s+//;
        $uid=~ s/\s+$//;
        return $uid;
    }
    error_log("find_uid: Could not extract UID from DN: $dn", 0);
    return undef;
}

sub find_upn{
	my $uid=shift;
	my $realm=shift;
	my $entry=shift;
	my $u_p_n= $entry->get_value('hrEduPersonUniqueID');
	
	unless($u_p_n){
		$u_p_n= $uid ? lc($uid . '@' . $realm . '.hr') : undef; # Construct UPN
	}
	return $u_p_n;
}


# Ensures Authorization header is set, refreshing token if needed
sub set_auth_header {
    my $reqRef = shift; # Reference to the HTTP::Request object
    # Check if token exists and hasn't expired (add buffer, e.g., 60 seconds)
    unless ($authToken && $authToken->{'expires_on'} && $authToken->{'expires_on'} > (time + 60)) {
        get_auth_token(); # Refresh token if invalid or expired
        unless ($authToken && $authToken->{access_token}) {
             # get_auth_token already logged the error and possibly died
             error_log("set_auth_header: Failed to obtain a valid auth token. Cannot proceed.", 1);
        }
    }
    # Set the header using the valid token
    $$reqRef->header(
        'Authorization' => $authToken->{'token_type'} . ' ' . $authToken->{'access_token'},
        'Content-Type'  => 'application/json; charset=utf-8'
        # 'Content-Transfer-Encoding'=>'binary' # Usually not needed for JSON/LWP
    );
}

# Fetches OAuth2 token from Microsoft Identity Platform (v2.0 endpoint)
sub get_auth_token {
    # print "Requesting new access token...\n";
    # Use LWP's post method for form data
    my $response=$ua->post($OauthTokenEndpoint,
        { # Use hash ref for form data
            'grant_type'    => 'client_credentials',
            'client_secret' => $clientSecret,
            'scope'         => $graphScope, # Use scope for MS Graph
            'client_id'     => $clientID
        }
        # No Content-Type needed here, LWP sets application/x-www-form-urlencoded
    );

    unless ($response->is_success) {
        error_log("get_auth_token: Failed to get token.\nStatus: " . $response->status_line . "\nResponse: " . $response->decoded_content, 1); # Critical error
        # error_log exits, so no return needed
    }

    my $token_data = try { decode_json($response->decoded_content) } catch {
         error_log("get_auth_token: Failed to decode token JSON.\nError: $_\nResponse: " . $response->content, 1);
    };

    unless ($token_data && $token_data->{access_token} && $token_data->{expires_in} && $token_data->{token_type}) {
         error_log("get_auth_token: Invalid token data received.\n" . Dumper($token_data), 1);
    }

    # Calculate expiry time (current time + expires_in seconds)
    $token_data->{'expires_on'} = time + $token_data->{expires_in};
    $authToken = $token_data; # Store the complete token hash
    # print "Access token obtained, expires around: " . scalar(localtime($authToken->{'expires_on'})) . "\n";
}


# Central function for calling Microsoft Graph API
sub call_graph_api {
    my ($http_method, $path, $code_ok, $hash_content_ref, $query_string) = @_;
    # $path should start with '/' e.g. '/users' or '/groups/{id}/members/$ref'

    my $url = $serviceRootURL . $path; # Base URL + path segment
    if ($query_string) {
        $url .= '?' . $query_string; # Add query string if provided (e.g., $filter)
    }

    my $req = HTTP::Request->new( $http_method => $url );
    set_auth_header(\$req); # Add valid Auth header

    my $content_json;
    if ($hash_content_ref && ref($hash_content_ref) eq 'HASH') {
        # Ensure UTF-8 encoding for JSON payload
        #$content_json = encode_json($hash_content_ref);
        $content_json =JSON::XS->new->utf8(0)->encode($hash_content_ref);
        $req->content($content_json);
        # Content-Type already set in set_auth_header
    }
    
	#print "\nRequest:\n";
	#print Dumper(\$req);

	# --- Make the API Call ---
    my $response=$ua->request($req);
    
    # --- Handle Response ---
    if ($response->code == $code_ok) {
        # Success! Check if there's content to return
        if (length($response->content // '') > 0) {
             my $decoded_response = try { decode_json($response->decoded_content) } catch {
                  error_log("call_graph_api: Success code $code_ok received for $http_method $path, but failed to decode JSON response.\nError: $_\nResponse: " . $response->content, 0);
                  return undef; # Return undef on decode error despite success code
             };
             return $decoded_response;
        } else {
            # Success, but no content (e.g., 204 No Content)
            return 1; # Return true value to indicate success
        }
    } else {
        # --- Error Handling ---
        my $error_details = $response->decoded_content;
        my $error_message = "call_graph_api: Error - $http_method $url\nExpected: $code_ok, Received: " . $response->status_line;
        if ($content_json) {
           $error_message .= "\nRequest Body: " . Dumper($hash_content_ref); # Dump original hash for readability
        }
         $error_message .= "\nResponse: " . $error_details;

         # Try to parse MS Graph error structure
         my $graph_error = try { decode_json($error_details)->{error} } catch { undef };
         if ($graph_error && $graph_error->{code} && $graph_error->{message}) {
             $error_message .= "\nParsed Error Code: " . $graph_error->{code};
             $error_message .= "\nParsed Error Message: " . $graph_error->{message};
         }

        error_log($error_message, 0); # Log error, don't die here, let calling function decide
        return 0; # Return false value to indicate failure
    }
}

# --- Utility Functions (mostly unchanged) ---

sub error_log {
    my $message=shift;
    my $umri=shift; # die if true

    #warn my_time() . " [ERROR] " . $message . "\n"; # Use warn to print to STDERR
    print my_time() . " [ERROR] " . $message . "\n";
    
    if ($umri) {
        print " Kriticna greska, izlazim.\n";
        die " Kriticna greska, izlazim.\n";
    }
}

sub makni_hrvatske {
    my $txt=shift // ''; # Handle undef input
    $txt =~ s/[^[:ascii:]]+//g;
    return $txt;
}

sub dump_content{
    my $content=shift;
    return Dumper(decode_json($content));
}

# Password generation - unchanged, but ensure it meets Azure AD complexity requirements
sub pwd_gen{
	my @a = split '', 'qwertyuiopasdfghjklzxcvbnm';
	my @b = split '', 'QWERTYUIOPASDFGHJKLZXCVBNM';
	my @c = split '', ',.?<>;:@/!%^&*()-+=_'; # Check if all these are allowed by Azure AD policy
	my @d = split '', '0123456789';
	# Ensure minimum length (e.g., 12 chars total matches example below)
	my @e = split '', 'qwertyuiopasdfghjklzxcvbnmQWERTYUIOPASDFGHJKLZXCVBNM0123456789'; # Removed special chars here for broader base
	my $pass=pick_chrs(\@a,3); # 3 lowercase
	$pass.=pick_chrs(\@b,3); # 3 uppercase
	$pass.=pick_chrs(\@c,2); # 2 special
	$pass.=pick_chrs(\@d,3); # 3 digits
	$pass.=pick_chrs(\@e,1); # 1 more from general pool (total 12)
	return join('',shuffle(split('',$pass)));
}
sub pick_chrs{
	my $c_ref=shift;
	my $num=shift;
	my @c=@{$c_ref};
    return '' unless @c; # Avoid errors if array is empty
	my $pass='';
	for(my $i=0;$i<$num;$i++){
		$pass .= $c[rand @c];
	}
	return $pass;
}

sub my_time {
	my ($second, $minute, $hour, $dayOfMonth, $month, $yearOffset) = localtime();
	my $year = 1900 + $yearOffset;
	$month++;
	return sprintf("%04d-%02d-%02d %02d:%02d:%02d", $year, $month, $dayOfMonth, $hour, $minute, $second);
}

sub timestamp {
	my ($second, $minute, $hour, $dayOfMonth, $month, $yearOffset) = localtime();
	my $year = 1900 + $yearOffset;
	$month++;
	return sprintf("%04d%02d%02d-%02d%02d%02d", $year, $month, $dayOfMonth, $hour, $minute, $second); # Changed format slightly for filename sorting
}
